<?php
session_start();

// --- НАСТРОЙКИ ФАЙЛОВОЙ СИСТЕМЫ ---
$DATA_FILE = 'guestbook_data.json';
// ---

// --- ФУНКЦИИ ДЛЯ РАБОТЫ С ФАЙЛОМ ---

/**
* Читает данные из JSON-файла. 
* Если файл не существует или пуст, возвращает начальную структуру.
*/
function readData($file) {
if (!file_exists($file) || filesize($file) === 0) {
return ['messages' => [], 'last_id' => 0];
}
// Используем flock для безопасного чтения
$handle = fopen($file, 'r');
if ($handle === false) {
throw new Exception("Не удалось открыть файл данных для чтения.");
}
flock($handle, LOCK_SH);
$content = stream_get_contents($handle);
flock($handle, LOCK_UN);
fclose($handle);

$data = json_decode($content, true);
return $data ?: ['messages' => [], 'last_id' => 0];
}

/**
* Записывает данные обратно в JSON-файл.
*/
function writeData($file, $data) {
$json_content = json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
// Используем flock для безопасной записи
$result = file_put_contents($file, $json_content, LOCK_EX);
if ($result === false) {
throw new Exception("Не удалось записать данные в файл.");
}
}

// --- ИНИЦИАЛИЗАЦИЯ ---
try {
$guestbook_data = readData($DATA_FILE);
} catch (Exception $e) {
$_SESSION['main_form_error'] = 'Критическая ошибка: ' . $e->getMessage();
$guestbook_data = ['messages' => [], 'last_id' => 0];
}

// Инициализация переменных формы
$name = '';
$email = '';
$city = '';
$website = '';
$message_content = '';
$captcha = '';

// --- ОБРАБОТКА ОТПРАВКИ ФОРМЫ (POST) (CLIENT) ---
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
$name = trim($_POST['name'] ?? '');
$email = trim($_POST['email'] ?? '');
$city = trim($_POST['city'] ?? '');
$website = trim($_POST['website'] ?? '');
$message_content = trim($_POST['message'] ?? '');
$captcha = $_POST['captcha'] ?? '';

$errors = [];

// Проверка капчи (всегда)
if (!isset($_SESSION['captcha']) || strtolower($captcha) !== strtolower($_SESSION['captcha'])) {
$errors[] = 'Неверный код с картинки.';
} else {
unset($_SESSION['captcha']);
}

// Проверка обязательных полей
if (empty($name)) {
$errors[] = 'Пожалуйста, введите Ваше имя.';
}

// Проверка сообщения и очистка от недопустимых тегов
if (empty($message_content)) {
$errors[] = 'Пожалуйста, введите текст сообщения.';
} else {
$allowed_tags = '<b><i><u><strong><em><a><br><span><p><ul><ol><li>';
$message_content = strip_tags($message_content, $allowed_tags);
// Блокировка потенциально опасных ссылок
$message_content = preg_replace('/<a([^>]*)href=["\'](javascript|data):/i', '<a$1href="blocked:' , $message_content);
}

// Проверка URL
if (!empty($website) && !filter_var($website, FILTER_VALIDATE_URL)) {
$errors[] = 'Указан некорректный URL сайта.';
}

if (empty($errors)) {
try {
// --- СОЗДАНИЕ НОВОГО (INSERT) (CLIENT) ---
$new_id = $guestbook_data['last_id'] + 1;

$new_message = [
'id' => $new_id,
'reply_to' => null,
'name' => $name,
'email' => $email,
'city' => $city,
'website' => $website,
'message' => $message_content,
'created_at' => date('Y-m-d H:i:s'),
'replies' => [],
];

// Добавляем новое сообщение в массив
// Порядок будет гарантирован сортировкой ниже
$guestbook_data['messages'][] = $new_message;
$guestbook_data['last_id'] = $new_id;

writeData($DATA_FILE, $guestbook_data);

$_SESSION['main_form_success'] = 'Сообщение успешно добавлено.';
header('Location: index.php'); // Ссылка на себя
exit;

} catch (Exception $e) {
$_SESSION['main_form_error'] = 'Ошибка при сохранении данных: ' . $e->getMessage();
$_SESSION['main_form_fields'] = [
'name' => $name,
'email' => $email,
'city' => $city,
'website' => $website,
'message' => $message_content,
];
header('Location: index.php'); // Ссылка на себя
exit;
}

} else {
// Обработка ошибок валидации
$error = implode('<br>', $errors);
$_SESSION['main_form_error'] = $error;
$_SESSION['main_form_fields'] = [
'name' => $name,
'email' => $email,
'city' => $city,
'website' => $website,
'message' => $message_content,
];
header('Location: index.php'); // Ссылка на себя
exit;
}
}

// --- ПОДГОТОВКА ПЕРЕМЕННЫХ ДЛЯ ШАБЛОНА И ПАГИНАЦИЯ ---
$main_form_success = isset($_SESSION['main_form_success']) ? $_SESSION['main_form_success'] : '';
unset($_SESSION['main_form_success']);

$main_form_error = isset($_SESSION['main_form_error']) ? $_SESSION['main_form_error'] : '';
unset($_SESSION['main_form_error']);

if (isset($_SESSION['main_form_fields'])) {
$main_form_fields = $_SESSION['main_form_fields'];
unset($_SESSION['main_form_fields']);
$name = $main_form_fields['name'] ?? $name;
$email = $main_form_fields['email'] ?? $email;
$city = $main_form_fields['city'] ?? $city;
$website = $main_form_fields['website'] ?? $website;
$message_content = $main_form_fields['message'] ?? $message_content;
}

// --- ПАГИНАЦИЯ ---
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
if ($page < 1) $page = 1;

$limit = 9;
$allMessages = $guestbook_data['messages'];

// СОРТИРОВКА: Новые сообщения сверху (DESC по дате)
usort($allMessages, function($a, $b) {
return strtotime($b['created_at']) - strtotime($a['created_at']);
});

$totalMessages = count($allMessages);
$totalPages = ceil($totalMessages / $limit);

if ($page > $totalPages && $totalPages > 0) {
$page = $totalPages;
} elseif ($totalMessages == 0) {
$page = 1;
}

$offset = ($page - 1) * $limit;
$mainMessages = array_slice($allMessages, $offset, $limit);
?>
<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.01 Transitional//EN">
<html>
<head>
<meta http-equiv='content-type' content='text/html; charset=utf-8'>
<meta http-equiv='content-language' content='ru'>
<title>Гостевая книга</title>
<link rel="icon" type="image/x-icon" href="/img/favicon.ico">
<link href="scripts/style.css" rel="stylesheet" type="text/css">
<style>
.pagination a.active-page {
color: #ff8c00; 
font-weight: bold;
}
</style>
<script type='text/javascript' src='scripts/jscript.js'></script>
<script type='text/javascript' src='scripts/js.js'></script>
</head>
<body>
<center>
<table width="350" align="center" border="1" class="main">
<tbody class="info">
<tr><td>
<center><b>Гостевая книга</b></center>
</td></tr>
<tr><td>
<table width='100%' cellspacing='0' cellpadding='0' class='board'>
<tr><td class='pages'>
<div class="pagination">
<?php if ($totalPages > 1): ?>
<b>Страница <?= $page ?> из <?= $totalPages ?>:</b>
<?php for ($i = 1; $i <= $totalPages; $i++): ?>
<a href="?page=<?= $i ?>" class="<?= ($i == $page) ? 'active-page' : '' ?>">[<?= $i ?>]</a>
<?php endfor; ?>
<?php endif; ?>
</div>
</tr></td>
</table><br>

<div class="messages">
<?php if (!empty($mainMessages)): ?>
<?php foreach ($mainMessages as $message): ?>
<table width='100%' cellspacing='0' cellpadding='0' class='board'>
<tr>
	<td class='msgtopuname'><strong><?= htmlspecialchars($message['name']) ?></strong></td>
	<td class='msgtopdate'><span><?= date('H:i, d.m.Y', strtotime($message['created_at'])) ?></span></td>
	<td class='msgtopip'>#<?= $message['id'] ?></td>
</tr>
<tr>
<td class='msgudata'>
	<?php if (!empty($message['website'])): ?>&nbsp;<img src="img/url.gif"> <a href="#tosite" onclick="return handleWebsiteClick(this);" data-website="<?= htmlspecialchars($message['website']) ?>" rel="nofollow noopener" target="_blank">Ссылка на сайт</a><br><?php endif; ?>
	<?php if (!empty($message['city'])): ?>&nbsp;<img src="img/city.gif"> <?= htmlspecialchars($message['city']) ?><?php endif; ?>
</td>
<td colspan='2' class='msgtext'>
<?= nl2br($message['message']) ?> <?php if (!empty($message['replies'])): ?>
<div class="replies">
<?php foreach ($message['replies'] as $reply): ?>
<div class="answer">
<span class='miniword'>Ответ:</span><br>
<?= nl2br($reply['message']) ?> </div>
<?php endforeach; ?></div>
<?php endif; ?>
</td></tr>
</table><br>

<?php endforeach; ?><?php else: ?><p>Пока нет сообщений.</p><?php endif; ?></div>

<table width='100%' cellspacing='0' cellpadding='0' class='board'><tr><td class='pages'>
<div class="pagination">
	<?php if ($totalPages > 1): ?><b>Страница <?= $page ?> из <?= $totalPages ?>:</b><?php for ($i = 1; $i <= $totalPages; $i++): ?>
	<a href="?page=<?= $i ?>" class="<?= ($i == $page) ? 'active-page' : '' ?>">[<?= $i ?>]</a><?php endfor; ?><?php endif; ?>
</div>
</tr></td></table><br>


<form method="POST" id="guestbook-form-id">
<table width='100%' cellspacing='0' cellpadding='2' class='board'>
<input type="hidden" name="reply_to" id="reply_to_field" value="">
<tr>
	<td>* Ваше имя:</td><td><input type="text" id="name" name="name" style="width: 200px; height: 20px;"
value="<?= htmlspecialchars($name) ?>" required></td>
	<td>Ваш сайт:</td><td><input type="url" id="website" name="website" style="width: 200px; height: 20px;"
value="<?= htmlspecialchars($website) ?>"></td>
</tr>
<tr>
	<td>E-mail:</td><td><input type="email" id="email" name="email" style="width: 200px; height: 20px;"
value="<?= htmlspecialchars($email) ?>"></td>
	<td>Город:</td><td><input type="text" id="city" name="city" style="width: 200px; height: 20px;"
value="<?= htmlspecialchars($city) ?>"></td>
</tr>
<tr>
	<td>* Введите текст сообщения:</td>
	<td colspan="3"><textarea id="message" name="message" style="width: 480px; height: 100px;" required><?= htmlspecialchars($message_content) ?></textarea></td>
</tr>
<tr>
	<td>* Код с картинки:</td>
	<td colspan="3"><img src="captcha.php?_t=<?= time() ?>" class="captcha-img" width='65' height='18'>
	<input type="text" name="captcha" style="width: 100px; height: 20px;">
</td></tr>
<tr>
	<td colspan="4" style="text-align: center;"><input type='submit' value='Добавить' class='button' />	
</td></tr>
</table>
</form>

</td></tr>
</tbody></table>
</center>
</body>
</html>