document.addEventListener('DOMContentLoaded', function() {
    const audioPlayer = document.getElementById('audioPlayer');
    const playBtn = document.getElementById('playBtn');
    const prevBtn = document.getElementById('prevBtn');
    const nextBtn = document.getElementById('nextBtn');
    const repeatBtn = document.getElementById('repeatBtn');
    const progressBar = document.getElementById('progressBar');
    const volumeSlider = document.getElementById('volumeSlider');
    const currentTimeEl = document.getElementById('currentTime');
    const durationEl = document.getElementById('duration');
    const nowPlayingEl = document.getElementById('nowPlaying');
    const contentList = document.getElementById('contentList');
    const breadcrumb = document.getElementById('breadcrumb');
    const searchInput = document.getElementById('searchInput');
    const searchButton = document.getElementById('searchButton');
    const playerContainer = document.querySelector('.player-container');
    
    let currentSongIndex = 0;
    let songs = [];
    let currentPath = 'music/';
    let isRepeatOn = false;
    
    // Инициализация
    initContent();
    
    // Обработчики для папок и песен
    document.addEventListener('click', function(e) {
        if (e.target.classList.contains('folder-item') || e.target.parentElement.classList.contains('folder-item')) {
            const folderItem = e.target.classList.contains('folder-item') ? e.target : e.target.parentElement;
            const path = folderItem.getAttribute('data-path');
            const backPath = folderItem.getAttribute('data-back');
            loadFolder(path, backPath);
            e.preventDefault();
        }
        
        if (e.target.classList.contains('song-item') || e.target.parentElement.classList.contains('song-item')) {
            const songItem = e.target.classList.contains('song-item') ? e.target : e.target.parentElement;
            const songIndex = Array.from(contentList.querySelectorAll('.song-item')).indexOf(songItem);
            if (songIndex !== -1) {
                playSong(songIndex);
            }
            e.preventDefault();
        }
        
        if (e.target.classList.contains('breadcrumb-item') || 
            (e.target.parentElement.classList.contains('breadcrumb-item') && e.target.tagName === 'A')) {
            const breadcrumbItem = e.target.classList.contains('breadcrumb-item') ? e.target : e.target.parentElement;
            const path = breadcrumbItem.querySelector('a')?.getAttribute('data-path') || 'music/';
            loadFolder(path);
            e.preventDefault();
        }
    });
    
    // Повтор трека
    repeatBtn.addEventListener('click', function() {
        isRepeatOn = !isRepeatOn;
        if (isRepeatOn) {
            repeatBtn.classList.add('active');
            repeatBtn.innerHTML = '<i class="fas fa-redo" style="color: #007790;"></i>';
        } else {
            repeatBtn.classList.remove('active');
            repeatBtn.innerHTML = '<i class="fas fa-redo"></i>';
        }
    });
    
    // Загрузка содержимого папки
    function loadFolder(path, backPath = '') {
        currentPath = path;
        
        // Очищаем список
        contentList.innerHTML = '';
        
        // Добавляем кнопку "Назад" если есть куда возвращаться
        if (backPath && backPath !== 'music/') {
            const backItem = document.createElement('a');
            backItem.href = '#';
            backItem.className = 'list-group-item list-group-item-action folder-item';
            backItem.innerHTML = '<i class="fas fa-arrow-left me-2"></i>Назад';
            backItem.setAttribute('data-path', backPath);
            backItem.setAttribute('data-back', getParentPath(backPath));
            contentList.appendChild(backItem);
        }
        
        // Получаем содержимое папки
        fetchFolderContent(path, backPath);
        updateBreadcrumb(path);
    }
    
    // Получение содержимого папки
    function fetchFolderContent(path, backPath) {
        const xhr = new XMLHttpRequest();
        xhr.open('GET', `get_folder_content.php?path=${encodeURIComponent(path)}&back=${encodeURIComponent(backPath)}`, true);
        xhr.onload = function() {
            if (this.status === 200) {
                contentList.innerHTML += this.responseText;
                initContent();
            }
        };
        xhr.send();
    }
    
    // Получение родительской папки
    function getParentPath(path) {
        const parts = path.split('/');
        parts.pop(); // Удаляем последний элемент
        parts.pop(); // Удаляем еще один, чтобы получить родительскую папку
        return parts.join('/') + '/';
    }
    
    // Обновление хлебных крошек
    function updateBreadcrumb(path) {
        breadcrumb.innerHTML = `
            <li class="breadcrumb-item">
                <a href="#" data-path="music/">Главная</a>
            </li>
        `;
        
        if (path && path !== 'music/') {
            const parts = path.replace('music/', '').split('/');
            let currentPath = 'music/';
            
            parts.forEach((part, index) => {
                if (part) {
                    currentPath += part + '/';
                    const isLast = index === parts.length - 2; // -2 потому что последний элемент пустой
                    
                    breadcrumb.innerHTML += `
                        <li class="breadcrumb-item">
                            ${!isLast 
                                ? `<a href="#" data-path="${currentPath}">${part}</a>`
                                : `<span>${part}</span>`
                            }
                        </li>
                    `;
                }
            });
        }
    }
    
    // Инициализация контента
    function initContent() {
        const songItems = contentList.querySelectorAll('.song-item');
        songs = Array.from(songItems).map(item => ({
            name: item.textContent,
            src: item.getAttribute('data-src'),
            element: item
        }));
    }
    
    // Воспроизведение песни
    function playSong(index) {
        if (index >= 0 && index < songs.length) {
            currentSongIndex = index;
            const song = songs[currentSongIndex];
            
            audioPlayer.src = song.src;
            audioPlayer.play();
            
            nowPlayingEl.textContent = song.name;
            playBtn.innerHTML = '<i class="fas fa-pause"></i>';
            playerContainer.classList.remove('d-none'); // Показываем плеер
            
            // Обновляем активный элемент в списке
            document.querySelectorAll('.song-item').forEach(item => item.classList.remove('active'));
            song.element.classList.add('active');
            
            // Прокручиваем к активной песне
            song.element.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
        }
    }
    
    // Обновление прогресса
    audioPlayer.addEventListener('timeupdate', function() {
        const currentTime = audioPlayer.currentTime;
        const duration = audioPlayer.duration;
        
        if (duration) {
            const progressPercent = (currentTime / duration) * 100;
            progressBar.style.width = progressPercent + '%';
            
            currentTimeEl.textContent = formatTime(currentTime);
            durationEl.textContent = formatTime(duration);
        }
    });
    
    // Клик по прогресс-бару
    progressBar.parentElement.addEventListener('click', function(e) {
        const width = this.clientWidth;
        const clickX = e.offsetX;
        const duration = audioPlayer.duration;
        
        audioPlayer.currentTime = (clickX / width) * duration;
    });
    
    // Форматирование времени
    function formatTime(seconds) {
        const minutes = Math.floor(seconds / 60);
        const secs = Math.floor(seconds % 60);
        return `${minutes}:${secs < 10 ? '0' : ''}${secs}`;
    }
    
    // Управление воспроизведением
    playBtn.addEventListener('click', function() {
        if (audioPlayer.paused) {
            if (audioPlayer.src) {
                audioPlayer.play();
                playBtn.innerHTML = '<i class="fas fa-pause"></i>';
            } else if (songs.length > 0) {
                playSong(0);
            }
        } else {
            audioPlayer.pause();
            playBtn.innerHTML = '<i class="fas fa-play"></i>';
        }
    });
    
    // Следующая песня
    nextBtn.addEventListener('click', function() {
        const nextIndex = (currentSongIndex + 1) % songs.length;
        playSong(nextIndex);
    });
    
    // Предыдущая песня
    prevBtn.addEventListener('click', function() {
        const prevIndex = (currentSongIndex - 1 + songs.length) % songs.length;
        playSong(prevIndex);
    });
    
    // Громкость
    volumeSlider.addEventListener('input', function() {
        audioPlayer.volume = this.value;
    });
    
    // Автопереход к следующей песне или повтор
    audioPlayer.addEventListener('ended', function() {
        if (isRepeatOn) {
            audioPlayer.currentTime = 0;
            audioPlayer.play();
        } else {
            nextBtn.click();
        }
    });
    
    // Поиск
    function searchContent() {
        const searchTerm = searchInput.value.toLowerCase();
        const items = contentList.querySelectorAll('.list-group-item');
        
        items.forEach(item => {
            const contentName = item.textContent.toLowerCase();
            if (contentName.includes(searchTerm)) {
                item.style.display = 'flex';
            } else {
                item.style.display = 'none';
            }
        });
    }
    
    searchInput.addEventListener('input', searchContent);
    searchButton.addEventListener('click', searchContent);
    
    // Загрузка с клавиатуры
    document.addEventListener('keydown', function(e) {
        if (e.code === 'Space') {
            e.preventDefault();
            playBtn.click();
        }
    });
});