# app.py
import os
import flask
from flask import Flask, render_template, request, jsonify, redirect, url_for, flash
from flask_sqlalchemy import SQLAlchemy
from flask_login import LoginManager, UserMixin, login_user, logout_user, login_required, current_user
from flask_cors import CORS
import io
import sys
import subprocess
import tempfile
import os
import uuid
from datetime import datetime
from werkzeug.security import generate_password_hash, check_password_hash
from typing import Optional

app = Flask(__name__)
CORS(app)  # Enable CORS for all routes
app.config['SECRET_KEY'] = 'your-secret-key-change-in-production'

# Database configuration
if os.environ.get('FLASK_ENV') == 'production':
    # Production database (PostgreSQL)
    app.config['SQLALCHEMY_DATABASE_URI'] = os.environ.get(
        'DATABASE_URL', 
        'postgresql://pyweb_user:pyweb_password@db:5432/pyweb_db'
    )
else:
    # Development database (SQLite)
    app.config['SQLALCHEMY_DATABASE_URI'] = 'sqlite:///app.db'

app.config['SQLALCHEMY_TRACK_MODIFICATIONS'] = False

db = SQLAlchemy(app)
login_manager = LoginManager()
login_manager.init_app(app)
login_manager.login_view = "login"  # type: ignore

# Models
class User(UserMixin, db.Model):
    id = db.Column(db.Integer, primary_key=True)
    username = db.Column(db.String(80), unique=True, nullable=False)
    password_hash = db.Column(db.String(120), nullable=False)
    notebooks = db.relationship('Notebook', backref='author', lazy=True)
    
    def set_password(self, password):
        self.password_hash = generate_password_hash(password)
        
    def check_password(self, password):
        return check_password_hash(self.password_hash, password)

class Notebook(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    title = db.Column(db.String(100), nullable=False)
    code = db.Column(db.Text, nullable=False)
    input_data = db.Column(db.Text, nullable=False, default='')
    created_at = db.Column(db.DateTime, nullable=False, default=datetime.utcnow)
    updated_at = db.Column(db.DateTime, nullable=False, default=datetime.utcnow, onupdate=datetime.utcnow)
    user_id = db.Column(db.Integer, db.ForeignKey('user.id'), nullable=False)
    is_public = db.Column(db.Boolean, default=False, nullable=False)

@login_manager.user_loader
def load_user(user_id):
    return User.query.get(int(user_id))

# Create tables
with app.app_context():
    db.create_all()

# Routes
@app.route('/')
def index():
    """Отображает главную страницу."""
    public_notebooks = Notebook.query.filter_by(is_public=True).order_by(Notebook.created_at.desc()).limit(10).all()
    return render_template('index.html', public_notebooks=public_notebooks)

@app.route('/register', methods=['GET', 'POST'])
def register():
    """Регистрация нового пользователя."""
    if request.method == 'POST':
        username = request.form['username']
        password = request.form['password']
        
        # Проверяем, существует ли пользователь
        if User.query.filter_by(username=username).first():
            flash('Пользователь с таким именем уже существует')
            return redirect(url_for('register'))
        
        # Создаем нового пользователя
        user = User()
        user.username = username
        user.set_password(password)
        db.session.add(user)
        db.session.commit()
        
        flash('Регистрация успешна')
        return redirect(url_for('login'))
    
    return render_template('register.html')

@app.route('/login', methods=['GET', 'POST'])
def login():
    """Вход пользователя."""
    if request.method == 'POST':
        username = request.form['username']
        password = request.form['password']
        
        user = User.query.filter_by(username=username).first()
        
        if user and user.check_password(password):
            login_user(user)
            return redirect(url_for('dashboard'))
        else:
            flash('Неправильное имя пользователя или пароль')
    
    return render_template('login.html')

@app.route('/logout')
@login_required
def logout():
    """Выход пользователя."""
    logout_user()
    return redirect(url_for('index'))

@app.route('/dashboard')
@login_required
def dashboard():
    """Личный кабинет пользователя."""
    notebooks = Notebook.query.filter_by(user_id=current_user.id).order_by(Notebook.updated_at.desc()).all()
    return render_template('dashboard.html', notebooks=notebooks)

@app.route('/notebook/new', methods=['GET', 'POST'])
@login_required
def new_notebook():
    """Создание нового блокнота."""
    if request.method == 'POST':
        title = request.form['title']
        code = request.form['code']
        input_data = request.form['input_data']
        is_public = 'is_public' in request.form
        
        notebook = Notebook(
            title=title,
            code=code,
            input_data=input_data,
            is_public=is_public,
            user_id=current_user.id
        )
        
        db.session.add(notebook)
        db.session.commit()
        
        return redirect(url_for('edit_notebook', notebook_id=notebook.id))
    
    return render_template('notebook_form.html', notebook=None)

@app.route('/notebook/<int:notebook_id>/edit', methods=['GET', 'POST'])
@login_required
def edit_notebook(notebook_id):
    """Редактирование блокнота."""
    notebook = Notebook.query.get_or_404(notebook_id)
    
    # Проверяем, что пользователь является владельцем блокнота
    if notebook.user_id != current_user.id:
        flash('У вас нет прав для редактирования этого блокнота')
        return redirect(url_for('dashboard'))
    
    if request.method == 'POST':
        notebook.title = request.form['title']
        notebook.code = request.form['code']
        notebook.input_data = request.form['input_data']
        notebook.is_public = 'is_public' in request.form
        notebook.updated_at = datetime.utcnow()
        
        db.session.commit()
        
        return redirect(url_for('edit_notebook', notebook_id=notebook.id))
    
    return render_template('notebook_form.html', notebook=notebook)

@app.route('/notebook/<int:notebook_id>/delete', methods=['POST'])
@login_required
def delete_notebook(notebook_id):
    """Удаление блокнота."""
    notebook = Notebook.query.get_or_404(notebook_id)
    
    # Проверяем, что пользователь является владельцем блокнота
    if notebook.user_id != current_user.id:
        flash('У вас нет прав для удаления этого блокнота')
        return redirect(url_for('dashboard'))
    
    db.session.delete(notebook)
    db.session.commit()
    
    flash('Блокнот удален')
    return redirect(url_for('dashboard'))

@app.route('/notebook/<int:notebook_id>')
def view_notebook(notebook_id):
    """Просмотр блокнота."""
    notebook = Notebook.query.get_or_404(notebook_id)
    
    # Если блокнот приватный, проверяем права доступа
    if not notebook.is_public and (not current_user.is_authenticated or notebook.user_id != current_user.id):
        flash('У вас нет прав для просмотра этого блокнота')
        return redirect(url_for('index'))
    
    return render_template('view_notebook.html', notebook=notebook)

@app.route('/notebook/<int:notebook_id>/embed')
def embed_notebook(notebook_id):
    """Встраиваемый просмотр блокнота."""
    notebook = Notebook.query.get_or_404(notebook_id)
    
    # Если блокнот приватный, проверяем права доступа
    if not notebook.is_public and (not current_user.is_authenticated or notebook.user_id != current_user.id):
        return "У вас нет прав для просмотра этого блокнота", 403
    
    return render_template('embed_notebook.html', notebook=notebook)

@app.route('/notebook/<int:notebook_id>/embed-code')
@login_required
def embed_code(notebook_id):
    """Страница с кодом для встраивания блокнота."""
    notebook = Notebook.query.get_or_404(notebook_id)
    
    # Проверяем, что пользователь является владельцем блокнота
    if notebook.user_id != current_user.id:
        flash('У вас нет прав для получения кода встраивания этого блокнота')
        return redirect(url_for('index'))
    
    return render_template('embed_code.html', notebook=notebook)

@app.route('/run_code', methods=['POST', 'OPTIONS'])
def run_code():
    """Обрабатывает AJAX-запрос, выполняет код Python в изолированном venv и возвращает результат."""
    
    # Handle CORS preflight request
    if request.method == 'OPTIONS':
        response = jsonify({'status': 'ok'})
        response.headers.add('Access-Control-Allow-Origin', '*')
        response.headers.add('Access-Control-Allow-Headers', 'Content-Type')
        response.headers.add('Access-Control-Allow-Methods', 'POST')
        return response
    
    # Получаем код и входные данные из запроса
    data = request.get_json()
    code = data.get('code', '')
    input_data = data.get('input', '')
    
    # Создаем временную директорию для изолированного окружения
    with tempfile.TemporaryDirectory() as temp_dir:
        # Создаем виртуальное окружение
        venv_path = os.path.join(temp_dir, 'venv')
        subprocess.run([sys.executable, '-m', 'venv', venv_path], check=True)
        
        # Определяем путь к интерпретатору в виртуальном окружении
        if os.name == 'nt':  # Windows
            python_executable = os.path.join(venv_path, 'Scripts', 'python.exe')
        else:  # Unix/Linux/macOS
            python_executable = os.path.join(venv_path, 'bin', 'python')
        
        # Создаем временный файл для кода
        code_file = os.path.join(temp_dir, f'code_{uuid.uuid4().hex}.py')
        with open(code_file, 'w', encoding='utf-8') as f:
            f.write(code)
        
        # Подготовка стандартного ввода
        input_file = os.path.join(temp_dir, 'input.txt')
        with open(input_file, 'w', encoding='utf-8') as f:
            f.write(input_data)
        
        # Выполняем код в изолированном виртуальном окружении
        try:
            # Запускаем код с перенаправленным вводом/выводом
            with open(input_file, 'r') as stdin_file, \
                 subprocess.Popen([python_executable, code_file], 
                                 stdin=stdin_file,
                                 stdout=subprocess.PIPE, 
                                 stderr=subprocess.PIPE,
                                 text=True) as proc:
                stdout, stderr = proc.communicate()
                
                if proc.returncode == 0:
                    execution_result = stdout
                else:
                    execution_result = f"Ошибка выполнения:\n{stderr}"
                    
        except Exception as e:
            # Захватываем ошибки
            execution_result = f"Ошибка:\n{e}"
        
    # Возвращаем результат фронтенду
    response = jsonify({'output': execution_result})
    response.headers.add('Access-Control-Allow-Origin', '*')
    return response

if __name__ == '__main__':
    # Установите debug=False для продакшена
    app.run(debug=True, host='0.0.0.0', port=5001)